<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This page prints a review of a particular quiz attempt
 *
 * It is used either by the student whose attempts this is, after the attempt,
 * or by a teacher reviewing another's attempt during or afterwards.
 *
 * @package   mod_quiz
 * @copyright 1999 onwards Martin Dougiamas  {@link http://moodle.com}
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


require_once(__DIR__ . '/../../config.php');
require_once($CFG->dirroot . '/mod/quiz/locallib.php');
require_once($CFG->dirroot . '/mod/quiz/report/reportlib.php');

$attemptid = required_param('attempt', PARAM_INT);
$page      = optional_param('page', 0, PARAM_INT);
$showall   = optional_param('showall', null, PARAM_BOOL);

$url = new moodle_url('/mod/quiz/review.php', array('attempt'=>$attemptid));
if ($page !== 0) {
    $url->param('page', $page);
} else if ($showall) {
    $url->param('showall', $showall);
}
$PAGE->set_url($url);

$attemptobj = quiz_attempt::create($attemptid);
$page = $attemptobj->force_page_number_into_range($page);

// Now we can validate the params better, re-genrate the page URL.
if ($showall === null) {
    $showall = $page == 0 && $attemptobj->get_default_show_all('review');
}
$PAGE->set_url($attemptobj->review_url(null, $page, $showall));

// Check login.
require_login($attemptobj->get_course(), false, $attemptobj->get_cm());
$attemptobj->check_review_capability();

// Create an object to manage all the other (non-roles) access rules.
$accessmanager = $attemptobj->get_access_manager(time());
$accessmanager->setup_attempt_page($PAGE);

$options = $attemptobj->get_display_options(true);

// Check permissions - warning there is similar code in reviewquestion.php and
// quiz_attempt::check_file_access. If you change on, change them all.
if ($attemptobj->is_own_attempt()) {
    if (!$attemptobj->is_finished()) {
        redirect($attemptobj->attempt_url(null, $page));

    } else if (!$options->attempt) {
        $accessmanager->back_to_view_page($PAGE->get_renderer('mod_quiz'),
                $attemptobj->cannot_review_message());
    }

} else if (!$attemptobj->is_review_allowed()) {
    throw new moodle_quiz_exception($attemptobj->get_quizobj(), 'noreviewattempt');
}

// Load the questions and states needed by this page.
if ($showall) {
    $questionids = $attemptobj->get_slots();
} else {
    $questionids = $attemptobj->get_slots($page);
}

// Save the flag states, if they are being changed.
if ($options->flags == question_display_options::EDITABLE && optional_param('savingflags', false,
        PARAM_BOOL)) {
    require_sesskey();
    $attemptobj->save_question_flags();
    redirect($attemptobj->review_url(null, $page, $showall));
}

// Work out appropriate title and whether blocks should be shown.
if ($attemptobj->is_own_preview()) {
    $strreviewtitle = get_string('reviewofpreview', 'quiz');
    navigation_node::override_active_url($attemptobj->start_attempt_url());

} else {
    $strreviewtitle = get_string('reviewofattempt', 'quiz', $attemptobj->get_attempt_number());
    if (empty($attemptobj->get_quiz()->showblocks) && !$attemptobj->is_preview_user()) {
        $PAGE->blocks->show_only_fake_blocks();
    }
}

// Set up the page header.
$headtags = $attemptobj->get_html_head_contributions($page, $showall);
$PAGE->set_title($attemptobj->get_quiz_name());
$PAGE->set_heading($attemptobj->get_course()->fullname);

// Summary table start. ============================================================================

// Work out some time-related things.
$attempt = $attemptobj->get_attempt();
$quiz = $attemptobj->get_quiz();
/*$overtime = 0;

if ($attempt->state == quiz_attempt::FINISHED) {
    if ($timetaken = ($attempt->timefinish - $attempt->timestart)) {
        if ($quiz->timelimit && $timetaken > ($quiz->timelimit + 60)) {
            $overtime = $timetaken - $quiz->timelimit;
            $overtime = format_time($overtime);
        }
        $timetaken = format_time($timetaken);
    } else {
        $timetaken = "-";
    }
} else {
    $timetaken = get_string('unfinished', 'quiz');
}

// Prepare summary informat about the whole attempt.
$summarydata = array();
if (!$attemptobj->get_quiz()->showuserpicture && $attemptobj->get_userid() != $USER->id) {
    // If showuserpicture is true, the picture is shown elsewhere, so don't repeat it.
    $student = $DB->get_record('user', array('id' => $attemptobj->get_userid()));
    $userpicture = new user_picture($student);
    $userpicture->courseid = $attemptobj->get_courseid();
    $summarydata['user'] = array(
        'title'   => $userpicture,
        'content' => new action_link(new moodle_url('/user/view.php', array(
                                'id' => $student->id, 'course' => $attemptobj->get_courseid())),
                          fullname($student, true)),
    );
}

if ($attemptobj->has_capability('mod/quiz:viewreports')) {
    $attemptlist = $attemptobj->links_to_other_attempts($attemptobj->review_url(null, $page,
            $showall));
    if ($attemptlist) {
        $summarydata['attemptlist'] = array(
            'title'   => get_string('attempts', 'quiz'),
            'content' => $attemptlist,
        );
    }
}

// Timing information.
$summarydata['startedon'] = array(
    'title'   => get_string('startedon', 'quiz'),
    'content' => userdate($attempt->timestart),
);

$summarydata['state'] = array(
    'title'   => get_string('attemptstate', 'quiz'),
    'content' => quiz_attempt::state_name($attempt->state),
);

if ($attempt->state == quiz_attempt::FINISHED) {
    $summarydata['completedon'] = array(
        'title'   => get_string('completedon', 'quiz'),
        'content' => userdate($attempt->timefinish),
    );
    $summarydata['timetaken'] = array(
        'title'   => get_string('timetaken', 'quiz'),
        'content' => $timetaken,
    );
}

if (!empty($overtime)) {
    $summarydata['overdue'] = array(
        'title'   => get_string('overdue', 'quiz'),
        'content' => $overtime,
    );
}

// Show marks (if the user is allowed to see marks at the moment).
$grade = quiz_rescale_grade($attempt->sumgrades, $quiz, false);
if ($options->marks >= question_display_options::MARK_AND_MAX && quiz_has_grades($quiz)) {

    if ($attempt->state != quiz_attempt::FINISHED) {
        // Cannot display grade.

    } else if (is_null($grade)) {
        $summarydata['grade'] = array(
            'title'   => get_string('grade', 'quiz'),
            'content' => quiz_format_grade($quiz, $grade),
        );

    } else {
        // Show raw marks only if they are different from the grade (like on the view page).
        if ($quiz->grade != $quiz->sumgrades) {
            $a = new stdClass();
            $a->grade = quiz_format_grade($quiz, $attempt->sumgrades);
            $a->maxgrade = quiz_format_grade($quiz, $quiz->sumgrades);
            $summarydata['marks'] = array(
                'title'   => get_string('marks', 'quiz'),
                'content' => get_string('outofshort', 'quiz', $a),
            );
        }

        // Now the scaled grade.
        $a = new stdClass();
        $a->grade = html_writer::tag('b', quiz_format_grade($quiz, $grade));
        $a->maxgrade = quiz_format_grade($quiz, $quiz->grade);
        if ($quiz->grade != 100) {
            $a->percent = html_writer::tag('b', format_float(
                    $attempt->sumgrades * 100 / $quiz->sumgrades, 0));
            $formattedgrade = get_string('outofpercent', 'quiz', $a);
        } else {
            $formattedgrade = get_string('outof', 'quiz', $a);
        }
        $summarydata['grade'] = array(
            'title'   => get_string('grade', 'quiz'),
            'content' => $formattedgrade,
        );
    }
}

// Any additional summary data from the behaviour.
$summarydata = array_merge($summarydata, $attemptobj->get_additional_summary_data($options));

// Feedback if there is any, and the user is allowed to see it now.
$feedback = $attemptobj->get_overall_feedback($grade);
if ($options->overallfeedback && $feedback) {
    $summarydata['feedback'] = array(
        'title'   => get_string('feedback', 'quiz'),
        'content' => $feedback,
    );
}

// Summary table end. ==============================================================================

if ($showall) {
    $slots = $attemptobj->get_slots();
    $lastpage = true;
} else {
    $slots = $attemptobj->get_slots($page);
    $lastpage = $attemptobj->is_last_page($page);
}
*/




//echo "<script> alert(" . $quiz->id . "); </script>";


$output = $PAGE->get_renderer('mod_quiz');


// Arrange for the navigation to be displayed.
//$navbc = $attemptobj->get_navigation_panel($output, 'quiz_review_nav_panel', $page, $showall);
//$regions = $PAGE->blocks->get_regions();
//$PAGE->blocks->add_fake_block($navbc, reset($regions));

//echo $output->review_page($attemptobj, $slots, $page, $showall, $lastpage, $options, $summarydata);

echo $OUTPUT->header();

function info($object)
{
    echo "<pre>";
    var_dump($object);
    echo "</pre>";
}

function get_quiz_name($answer)
{
	$name = explode("]", $answer)[0];
	return substr($name, 1, strlen($name)-1);
}

function get_competence_name($answer)
{
	$name = explode("]", $answer)[1];
	return substr($name, 1, strlen($name)-1);
}

$user_answers = $DB->get_records_sql('SELECT (@row:=@row+1) AS row_id, quizid, userid, name, responsesummary FROM (SELECT @row:=0) AS rows, (SELECT DISTINCT qz.id AS quizid, qas.userid, q.name, qatt.responsesummary FROM {question} q INNER JOIN {question_attempts} qatt ON q.id = qatt.questionid INNER JOIN {question_attempt_steps} qas ON qatt.id = qas.questionattemptid INNER JOIN {quiz_slots} qs ON q.id = qs.questionid INNER JOIN {quiz} qz ON qs.quizid = qz.id WHERE quizid = ' . $quiz->id . ' AND userid = ' . $USER->id . ' ORDER BY qatt.timemodified DESC) x GROUP BY name', array());

$quiz_name = get_quiz_name($user_answers[1]->name);

$competences = array();
$score = array();

foreach ($user_answers as $user_answer) 
{
	if(strcmp(get_quiz_name($user_answer->name), "CSV") != 0)
	{
		$competences[get_competence_name($user_answer->name)][$user_answer->name] = explode(";", $user_answer->responsesummary);
		for ($i = 0; $i < count($competences[get_competence_name($user_answer->name)][$user_answer->name]); $i++) 
		{
			$competences[get_competence_name($user_answer->name)][$user_answer->name][$i] = trim($competences[get_competence_name($user_answer->name)][$user_answer->name][$i]);
		}
		$score[get_competence_name($user_answer->name)] = array("get" => 0, "max" => 0);
	}
}

foreach ($competences as $competence => $competence_answers) 
{

	$answers = $DB->get_records_sql('SELECT (@row:=@row+1) AS row_id, qz.id, q.name, qn.answer, qn.fraction FROM (SELECT @row:=0) AS rows, {question} q INNER JOIN {quiz_slots} qs ON q.id = qs.questionid INNER JOIN {quiz} qz ON qs.quizid = qz.id INNER JOIN {quiz_attempts} qa ON qz.id = qa.quiz INNER JOIN {question_answers} qn ON q.id = qn.question WHERE qz.id = ' . $quiz->id . ' AND qa.userid = ' . $USER->id . ' AND q.name LIKE "[' . $quiz_name . '][' . $competence . ']%"', array());

	foreach ($answers as $answer) 
	{
		foreach ($competence_answers as $q => $a) 
		{
			if($answer->name == $q)
			{
				foreach ($a as $answ) 
				{
					if(strcmp(strip_tags(trim($answer->answer)), $answ) == 0)
					{
						$score[$competence]["get"] += floatval($answer->fraction) > 0 ? 1 : 0;
					}
				}
			}
		}
		$score[$competence]["max"] += floatval($answer->fraction) > 0 ? 1 : 0;
	}
}


/*info($user_answers);
echo "--------------------------------------------------------------------------";
info($score);
info($answers);*/

function get_learn_type($csv)
{
	/*
		2 = KI
		6 = VI
		10 = AV
		http://prst-patterns.com/php/php_csv_export.php?glava=No;Ko;Ki;Ek;Em;Ae;Vi;Ak;Ra;In;Av;Ro&noga=27.000;42.000;38.000;19.000;32.000;55.000;20.000;58.000;41.000;81.000;42.000;45.000
		//27.000;42.000;38.000;19.000;32.000;55.000;20.000;58.000;41.000;81.000;42.000;45.000
	*/
	$values = explode(";", $csv);
	if(intval($values[2]) > intval($values[6]))
		if (intval($values[2]) > intval($values[10])) 
			return "KI";
		else
			return "AV";
	else
		if(intval($values[6]) > intval($values[10]))
			return "VI";
		else
			return "AV";
}

function get_score_text($score)
{
	if($score["get"]/$score["max"] > 0.9)
		return "odlična";

	if($score["get"]/$score["max"] > 0.7)
		return "ustrezna";

	if($score["get"]/$score["max"] > 0.5)
		return "minimalna";

	if($score["get"]/$score["max"] > 0.3)
		return "neustrezna";

	return "kritična";
}

function alert($text)
{
	echo "<SCRIPT>alert('" . $text . "');</SCRIPT>";
}

$learn_type_query = $DB->get_records_sql('SELECT (@row:=@row+1) AS row_id, qa.responsesummary
FROM (SELECT @row:=0) AS rows, {question} q INNER JOIN {question_attempts} qa ON q.id = qa.questionid INNER JOIN {question_attempt_steps} qs ON qa.id = qs.questionattemptid
WHERE q.name LIKE "[CSV]%" AND qs.userid = ' . $USER->id, array());

$learn_type = get_learn_type(explode("=", $learn_type_query[1]->responsesummary)[2]);

echo "<table class='generaltable'>";
echo "<tr>";
echo "<td>Kompetenca</td>";
echo "<td>Stopnja usposobljenosti</td>";
echo "<td>Gradivo</td>";
echo "</tr>";

// competence => score
foreach ($score as $c => $v) 
{
	$link = $DB->get_records_sql('SELECT (@row:=@row+1) AS row_id, q.name, qn.answer FROM (SELECT @row:=0) AS rows, {question} q INNER JOIN {question_answers} qn ON q.id = qn.question WHERE q.name LIKE "[' . $quiz_name . '][' . $c . '][links]%" AND qn.answer LIKE "%[' . $learn_type . ']%"', array());

	echo "<tr>";
	echo "<td>" . $c . "</td>";
	echo "<td>" . get_score_text($v) . "</td>";
	echo "<td><a href='". explode("]", strip_tags(trim($link[1]->answer)))[1] . "' target='_blank'>Link</a></td>";
	echo "</tr>";

	//echo "Kompetenca: " . $c . "<br>Stopnja usposobljenosti: " . get_score_text($v) . "<br>Gradivo: <a href='" . explode("]", strip_tags(trim($link[1]->answer)))[1] . "' target='_blank'>Link</a><br><br>";
}

echo "</table>";

//info($score);
//echo $learn_type . "<br>";
//info($learn_type_query);

// Trigger an event for this review.
$attemptobj->fire_attempt_reviewed_event();